/*
 * $Id$
 *
 * Serial Wire Debug Open Framework for ARM Cortex CPU, version 0.0.1.
 * Copyright (C) 2010 Tomasz Boleslaw CEDRO (http://www.tomek.cedro.info)
 *
 * License will be probably BSD
 *
 * Written by Tomasz Boleslaw CEDRO <tomek.cedro@gmail.com>, 2010;
 *
 */

//Uncomment this to build with main() and test program.
#define __SWDHAVEMAIN__
#define __SWDDEBUG__

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#if defined(__SWDDEBUG__) || defined(__SWDHAVEMAIN__) 
#include <stdio.h>
#endif

/* SWD Packets Bit Fields and Values */
#define SWD_REQ_START_BITNUM  7  // Packet Start bit, always 1
#define SWD_REQ_APnDP_BITNUM  6  // Access Port (high) or Debug Port (low) access
#define SWD_REQ_RnW_BITNUM    5  // Read (high) or Write (low) access
#define SWD_REQ_A2_BITNUM     4  // Register Address bit 2
#define SWD_REQ_A3_BITNUM     3  // Register Address bit 3
#define SWD_REQ_PARITY_BITNUM 2  // Odd Parity calculated from APnDP, RnW, A[2:3]
#define SWD_REQ_STOP_BITNUM   1  // Packet Stop bit, always 0
#define SWD_REQ_PARK_BITNUM   0  // Park wire and switch between receive/transmit

#define SWD_REQ_START_VAL  1  // Start Bit is always 1
#define SWD_REQ_STOP_VAL   0  // Stop Bit is always 0
#define SWD_REQ_PARK_VAL   1  // Park bus and put outputs into Hi-Z state

#define SWD_ACK_BITLEN     3
#define SWD_ACK_OK         0b100
#define SWD_ACK_WAIT       0b010
#define SWD_ACK_FAULT      0b001

#define SWD_DP_ADDR_IDCODE   0b00 // RO
#define SWD_DP_ADDR_ABORT    0b00 // WO
#define SWD_DP_ADDR_CRTLSTAT 0b01 // R/W, CTRLSEL=b0
#define SWD_DP_ADDR_WCR      0b01 // R/W, CTRLSEL=b1
#define SWD_DP_ADDR_RESEND   0b10 // RO
#define SWD_DP_ADDR_SELECT   0b10 // WO
#define SWD_DP_ADDR_RDBUF    0b11 // RO

/* SW-DP ABORT Register Bitmask */
#define SWD_DAPABORT_BITNUM   0
#define SWD_STKCMPCLR_BITNUM  1
#define SWD_STKERRCLR_BITNUM  2
#define SWD_WDERRCLR_BITNUM   3
#define SWD_ORUNERRCLR_BITNUM 4

/* SW-DP CTRL/STAT Register Bitmask and Control Values */
#define SWD_ORUNDETECT_BITNUM    0
#define SWD_STICKYORUN_BITNUM    1
#define SWD_TRNMODE_BITNUM       2
#define SWD_STICKYCMP_BITNUM     4
#define SWD_STICKYERR_BITNUM     5
#define SWD_READOK_BITNUM        6
#define SWD_WDATAERR_BITNUM      7
#define SWD_MASKLANE_BITNUM      8
#define SWD_TRNCNT_BITNUM        12
#define SWD_CDBGRSTREQ_BITNUM    26
#define SWD_CDBGRSTACK_BITNUM    27
#define SWD_CDBGPWRUPREQ_BITNUM  28
#define SWD_CDBGPWRUPACK_BITNUM  29
#define SWD_CSYSPWRUPREQ_BITNUM  30
#define SWD_CSYSPWRUPACK_BITNUM  31

#define SWD_MASKLANE_0 0b0001  // Compare byte lane 0 (0x------FF)
#define SWD_MASKLANE_1 0b0010  // Compare byte lane 1 (0x----FF--)
#define SWD_MASKLANE_2 0b0100  // Compare byte lane 2 (0x--FF----)
#define SWD_MASKLANE_3 0b1000  // Compare byte lane 3 (0xFF------)

/* SW-DP SELECT Register Bitmask */
#define SWD_CTRLSEL   0
#define SWD_APBANKSEL 4 
#define SWD_APSEL     24

/* SW-DP WCR Register Bitmask and Control Values */
#define SWD_PRESCALER  0
#define SWD_WIREMODE   6
#define SWD_TURNROUND  8

#define SWD_TURNROUND_1 0b00
#define SWD_TURNROUND_2 0b01
#define SWD_TURNROUND_3 0b10
#define SWD_TURNROUND_4 0b11

/* AHB-AP Registers Addresses */
#define AHB_AP_CONTROLSTATUS 0x00  // R/W, 32bit, reset value: 0x43800042 
#define AHB_AP_TAR           0x04  // R/W, 32bit, reset value: 0x00000000
#define AHB_AP_DRW           0x0C  // R/W, 32bit
#define AHB_AP_BD0           0x10  // R/W, 32bit
#define AHB_AP_BD1           0x14  // R/W, 32bit
#define AHB_AP_BD2           0x18  // R/W, 32bit
#define AHB_AP_BD3           0x1C  // R/W, 32bit
#define AHB_AP_DROMT         0xF8  // RO, 32bit, reset value: 0xE00FF000
#define AHB_AP_IDR           0xFC  // RO, 32bit, reset value: 0x24770001

/* Status and Error definitions */

#define SWD_OK 0
#define SWD_ERROR             -1
#define SWD_ERROR_NULLPOINTER -2
#define SWD_ERROR_NULLQUEUE   -3
#define SWD_ERROR_NULLTRN     -4
#define SWD_ERROR_BADPARAM    -5
#define SWD_ERROR_OUTOFMEM    -6
#define SWD_ERROR_BADRESULT   -7
#define SWD_ERROR_OUTOFRANGE  -8

#define SWD_DATA_MAXBITCOUNT   32
#define SWD_DATA_BYTESIZE      8

#define enum SWD_SWJ_MODE {
 SWD_SWJ_MODE_UNKNOWN=0,
 SWD_SWJ_MODE_JTAG,
 SWD_SWJ_MODE_SWD
}

/* SWD context keeps all information about the target status, including data and instruction registers, also the command queue.
 * Writes and reads from the target should be made only with use of this context, otherwise you will loose data synchronization with the target.
 * If you only want to use this SWD API for bitstream generation you can ommit the context operations, however it was created for easier SWD implementation in existing software such as UrJTAG or OpenOCD.
typedef {

} swd_context;

/* These functions pointers use cable driver to switch buffers into Tx/Rx.*/
int (*swd_trn_hosttx)();
int (*swd_trn_hostrx)();

#define SWD_CMDTYPE_READ  1
#define SWD_CMDTYPE_WRITE 2
#define SWD_CMDTYPE_TRN   3
#define SWD_CMDTYPE_ACK   4

typedef struct swd_cmd {
 union {          // payload data union
  int (*trn)();   // turnaround command function pointer
  char request;   // request header data
  char ack;       // acknowledge response from target
  int rdata;      // data read from target
  int wdata;      // data written to target
  char parity;    // parity bit for data
 };
 char bits;          // payload bit count = clk pulses
 char type;       // operation type as defined by SWD_CMDTYPE_*
 char done;       // non zero if operation result is present
 struct swd_cmd *prev, *next;
} swd_cmd;


/* DATA GENERATED BELOW ARE TO BE SHIFTED OUT MSB FIRST, UNLIKE IN JTAG!
 * These functions generate a bistream packets for SWD communications.
 * Bistream is returned in byte array pointed by *buffer parameter.
 * Actual positive byte count is returned by the function.
 * Negative return value means an error. Zero means no data was generated.
 */


swd_bitgen_reset(char *buffer){
    char rststr[8] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
    return sizeof(memcpy(buffer, rststr, sizeof(rststr)));
}

int swd_bitgen_switch_jtag_to_swd(char *buffer){
    char jtag2swd_seq[2] = {0xE7, 0x9E};

    return sizeof(memcpy(buffer, jtag2swd_seq, sizeof(jtag2swd_seq)));
}

/* BitSwap helper function that reverse bit order in *data. 
 * Least Significant Bit becomes Most Significant Bit.
 * Rersult contains only "count" bits for use with char or int easily.
 */
int swd_bitswap8(unsigned char *buffer, int bitcount){
 if (bitcount>8) return SWD_ERROR_BADPARAM;
 unsigned char bit, res=0; //res must be unsigned for proper shifting result
 #ifdef __SWDDEBUG__
 printf("|SWD_DEBUG: swd_bitswap8(%02X, %d);\n", *buffer, bitcount);
 #endif
 for (bit=0;bit<bitcount;bit++) {
  res=(res<<1)|(((*buffer>>bit)&1)?1:0); 
  #ifdef __SWDDEBUG__
  printf("|SWD_DEBUG: swd_bitswap8: in=%02X out=%02X bit=%d\n", *buffer, res, bit);
  #endif
 }
 *buffer=res;
 return SWD_OK;
}

int swd_bitswap32(unsigned int *buffer, int bitcount){
 if (bitcount>32) return SWD_ERROR_BADPARAM;
 unsigned int bit, res=0; //res must be unsigned for proper shifting result
 #ifdef __SWDDEBUG__
 printf("|SWD_DEBUG: swd_bitswap32(%08X, %d);\n", *buffer, bitcount);
 #endif
 for (bit=0;bit<bitcount;bit++) {
  res=(res<<1)|(((*buffer>>bit)&1)?1:0); 
  #ifdef __SWDDEBUG__
  printf("|SWD_DEBUG: swd_bitswap32: in=%08X out=%08X bit=%d\n", *buffer, res, bit);
  #endif
 }
 *buffer=(int)res;
 return SWD_OK;
}


/* Command Queue helper functions */

/* Append element pointed by *cmd at the end of the quque pointed by *cmdq. */
int swd_cmd_queue_append(swd_cmd *cmdq, swd_cmd *cmd){
 if (cmdq==NULL || cmd==NULL) return SWD_ERROR_NULLPOINTER;
 cmdq->next=cmd;
 cmd->prev=cmdq;
 cmdq=cmd; 
 return 1;
}

/* Find first element of a queue pointed by *cmdq */
swd_cmd* swd_cmd_queue_find_root(swd_cmd *cmdq){
    swd_cmd *cmd=cmdq;
    while (cmd->prev!=NULL) cmd=swd_cmd_queue_find_root(cmd);
    return cmd;
}

/* Free memory and destroy queue pointed by the *cmdq. */
void swd_cmd_queue_free(swd_cmd *cmdq){
    swd_cmd *cmd, *nextcmd;
    cmd=swd_cmd_queue_find_root(cmdq);
    while (cmd!=NULL) {
        nextcmd=cmd->next;
        free(cmd);
        cmd=nextcmd;
    }
}

/* Free memory and destroy queue head starting from its root up to the element pointed by *cmdq. */
void swd_cmd_queue_free_head(swd_cmd *cmdq){
    swd_cmd *cmdqroot, *nextcmdqroot;
    cmdqroot=swd_cmd_queue_find_root(cmdq);
    while(cmdqroot!=cmdq){
        nextcmdqroot=cmdqroot;
        free(cmdqroot);
        cmdqroot=nextcmdqroot;
    }
}




/* Below functions generate SWD Commands Queue Elements */

int swd_cmd_queue_append_request(swd_cmd *cmdq, char request){
 if (!cmdq) return SWD_ERROR_NULLQUEUE;
  swd_cmd *cmd;
  cmd=(swd_cmd *)calloc(1,sizeof(swd_cmd));
  cmd->request=request;
  cmd->bits=8;
  cmd->type=SWD_CMDTYPE_WRITE;
  swd_cmd_queue_append(cmdq, cmd);
} 

int swd_cmd_queue_append_trn(swd_cmd *cmdq, void *trnfp){
 if (cmdq==NULL || trnfp==NULL) return SWD_ERROR_NULLPOINTER;
  swd_cmd *cmd;
  cmd=(swd_cmd *)calloc(1,sizeof(swd_cmd));
  cmd->trn=trnfp;
  cmd->bits=1;    // TODO: put actual context setting from the WCR
  cmd->type=SWD_CMDTYPE_TRN;
  swd_cmd_queue_append(cmdq, cmd);
  return 1;
}

int swd_cmd_queue_append_read(swd_cmd *cmdq, int *data){
 if (cmdq==NULL || data==NULL) return SWD_ERROR_NULLPOINTER;
 swd_cmd *cmd;
 cmd=(swd_cmd *)calloc(1,sizeof(swd_cmd));
 if (cmd==NULL) return SWD_ERROR_OUTOFMEM;
 data=&cmd->rdata;  // ???
 cmd->bits=33;
 cmd->type=SWD_CMDTYPE_READ;
 swd_cmd_queue_append(cmdq, cmd);
 return 1;
}

int swd_cmd_queue_append_nread(swd_cmd *cmdq, int *data, int count){
 if (cmdq==NULL || data==NULL || count<1) return SWD_ERROR_BADPARAM;
 int i;
 for (i=0;i<count;i++) if (swd_cmd_queue_append_read(cmdq, &data[i])<1) return SWD_ERROR_BADRESULT;
 return i;
}

int swd_cmd_queue_append_write(swd_cmd *cmdq, int *data){
 if (cmdq==NULL || data==NULL) return SWD_ERROR_NULLPOINTER;
 swd_cmd *cmd;
 cmd=(swd_cmd *)calloc(1,sizeof(swd_cmd));
 if (cmd==NULL) return SWD_ERROR_OUTOFMEM;
 cmd->wdata=*data;  // ???
 cmd->bits=32;
 cmd->type=SWD_CMDTYPE_WRITE;
 swd_cmd_queue_append(cmdq, cmd);
 return 1;
}

int swd_cmd_queue_append_nwrite(swd_cmd *cmdq, int *data, int count){
 if (cmdq==NULL || data==NULL || count<1) return SWD_ERROR_BADPARAM;
  int i;
 for (i=0;i<count;i++) if (swd_cmd_queue_append_write(cmdq, &data[i])<1) return SWD_ERROR_BADRESULT;
 return i;
}

int swd_cmd_queue_append_ack(swd_cmd *cmdq){
 if (cmdq==NULL) return SWD_ERROR_NULLQUEUE;
 swd_cmd *cmd;
 cmd=(swd_cmd *)calloc(1,sizeof(swd_cmd)); 
 if (cmd==NULL) return SWD_ERROR_OUTOFMEM;  
 cmd->bits=3;
 cmd->type=SWD_CMDTYPE_ACK;
 return swd_cmd_queue_append(cmdq, cmd);
}

int swd_cmd_queue_append_trn_hosttx(swd_cmd *cmdq){
 return swd_cmd_queue_append_trn(cmdq, swd_trn_hosttx);
}

int swd_cmd_queue_append_trn_hostrx(swd_cmd *cmdq){
 return swd_cmd_queue_append_trn(cmdq, swd_trn_hostrx);
}

int swd_cmd_dpreset(swd_cmd *cmdq){
 char rststr[8] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
 int i;
 swd_cmd_queue_append_trn_hosttx(cmdq);
 for (i=0; i<sizeof(rststr);i++) swd_cmd_queue_append_request(cmdq, rststr[i]);
 return i;
}

int swd_cmd_jtag2swd(swd_cmd *cmdq){
 char jtag2swdseq[2] = {0x79, 0xE7};
 int i;
 swd_cmd_queue_append_trn_hosttx(cmdq);
 for (i=0; i<sizeof(jtag2swdseq);i++) swd_cmd_queue_append_request(cmdq, jtag2swdseq[i]);
 return i;
}



#ifdef __SWDHAVEMAIN__

int main(int argc, char argv[]){
 printf("Hello to SWD world, a simple SWD API test program by Tomek Cedro (http://www.tomek.cedro.info)\n");

 int a=0x01a0cede;
 unsigned char b=0x6F;
 printf("\nBitswap test:\n");
 printf("a size: %d bits, b size: %d bits\n", sizeof(a)*SWD_DATA_BYTESIZE, sizeof(b)*SWD_DATA_BYTESIZE);
 printf("Before bitswap: a=%08X , b=%02X \n", a, b);
 swd_bitswap32(&a, sizeof(a)*SWD_DATA_BYTESIZE);
 swd_bitswap8(&b, SWD_DATA_BYTESIZE);
 printf("After bitswap : a=%08X , b=%02X \n\n", a, b);
 return 0;
}
#endif
